<?php
// admin.php - Modern Dashboard

require_once 'config.php';
require_once 'layout.php';
require_once 'application_handler.php';

session_start();

$error_message = '';
$action_message = '';

// Auth Logic
if (isset($_POST['login'])) {
    if ($_POST['username'] === ADMIN_USERNAME && $_POST['password'] === ADMIN_PASSWORD) {
        $_SESSION['admin_logged_in'] = true;
    } else {
        $error_message = 'Invalid credentials provided.';
    }
}
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: admin.php');
    exit;
}
if (isset($_POST['app_id'], $_POST['action']) && isset($_SESSION['admin_logged_in'])) {
    $result = handle_application_action($_POST['app_id'], $_POST['action'], $_POST['reason'] ?? null);
    $result['success'] ? $action_message = $result['message'] : $error_message = $result['message'];
}

// Login Screen
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    render_header("Admin Login", 'public');
    ?>
    <div class="min-h-[90vh] flex items-center justify-center relative z-20">
        <div class="w-full max-w-md animate-slide-up">
            <div class="text-center mb-10">
                <div class="inline-block p-4 rounded-full bg-dark-800 border border-dark-700 shadow-2xl mb-4">
                    <i class="fa-solid fa-fingerprint text-4xl text-primary-500"></i>
                </div>
                <h1 class="text-4xl font-bold text-white tracking-tight">Access Control</h1>
            </div>
            
            <div class="glass-panel p-8 rounded-3xl shadow-2xl relative overflow-hidden">
                <?php if ($error_message): ?>
                    <div class="mb-6 bg-red-500/10 border border-red-500/50 text-red-200 p-4 rounded-xl flex items-center text-sm">
                        <i class="fa-solid fa-circle-exclamation mr-3 text-lg"></i> <?php echo htmlspecialchars($error_message); ?>
                    </div>
                <?php endif; ?>

                <form method="POST" action="admin.php" class="space-y-5">
                    <div class="space-y-1">
                        <label class="text-xs font-bold text-gray-500 uppercase ml-1">Username</label>
                        <input type="text" name="username" required class="w-full bg-dark-900/50 border border-dark-700 text-white rounded-xl px-4 py-3.5 focus:border-primary-500 focus:ring-1 focus:ring-primary-500 outline-none transition">
                    </div>
                    <div class="space-y-1">
                        <label class="text-xs font-bold text-gray-500 uppercase ml-1">Password</label>
                        <input type="password" name="password" required class="w-full bg-dark-900/50 border border-dark-700 text-white rounded-xl px-4 py-3.5 focus:border-primary-500 focus:ring-1 focus:ring-primary-500 outline-none transition">
                    </div>
                    <button type="submit" name="login" class="w-full bg-primary-600 hover:bg-primary-500 text-white font-bold py-4 rounded-xl shadow-lg shadow-primary-500/25 transition-all transform hover:-translate-y-0.5 mt-2">
                        Authenticate
                    </button>
                </form>
            </div>
        </div>
    </div>
    <?php
    render_footer();
    exit;
}

// Dashboard Logic
$applications = get_applications();
$pending_apps = [];
$stats = ['pending' => 0, 'interview' => 0, 'accepted' => 0, 'declined' => 0];

foreach ($applications as $app) {
    if ($app['status'] === 'pending') $pending_apps[] = $app;
    $status_key = $app['status'] === 'waiting_for_interview' ? 'interview' : $app['status'];
    if (isset($stats[$status_key])) $stats[$status_key]++;
}

render_header("Dashboard", 'dashboard');
?>

<!-- Welcome Section -->
<div class="flex flex-col lg:flex-row justify-between items-start lg:items-center mb-10 gap-6 animate-fade-in">
    <div>
        <h2 class="text-4xl font-bold text-white tracking-tight">Dashboard Overview</h2>
        <p class="text-gray-400 mt-2 text-lg">System Status: <span class="text-green-400 font-mono">ONLINE</span></p>
    </div>
    <div class="flex gap-3">
        <a href="application_form.php" target="_blank" class="px-5 py-2.5 bg-dark-800 hover:bg-dark-700 text-gray-300 rounded-xl border border-dark-700 transition font-medium">
            <i class="fa-solid fa-link mr-2"></i> Public Link
        </a>
        <button onclick="location.reload()" class="px-5 py-2.5 bg-primary-600 hover:bg-primary-500 text-white rounded-xl shadow-lg shadow-primary-500/20 transition font-bold">
            <i class="fa-solid fa-rotate mr-2"></i> Refresh Data
        </button>
    </div>
</div>

<!-- Stats Grid -->
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-10 animate-slide-up">
    <!-- Stat 1 -->
    <div class="glass-panel p-6 rounded-2xl relative overflow-hidden group">
        <div class="flex justify-between items-start z-10 relative">
            <div>
                <p class="text-gray-400 text-xs font-bold uppercase tracking-widest">Pending</p>
                <h3 class="text-3xl font-bold text-white mt-2"><?php echo $stats['pending']; ?></h3>
            </div>
            <div class="p-3 bg-yellow-500/10 rounded-lg text-yellow-500">
                <i class="fa-solid fa-clock text-xl"></i>
            </div>
        </div>
        <div class="absolute bottom-0 left-0 h-1 bg-yellow-500 transition-all duration-1000" style="width: <?php echo ($stats['pending'] > 0) ? '100%' : '5%'; ?>"></div>
    </div>

    <!-- Stat 2 -->
    <a href="interview_queue.php" class="glass-panel p-6 rounded-2xl relative overflow-hidden group hover:bg-dark-800 transition">
        <div class="flex justify-between items-start z-10 relative">
            <div>
                <p class="text-gray-400 text-xs font-bold uppercase tracking-widest">Interviews</p>
                <h3 class="text-3xl font-bold text-white mt-2"><?php echo $stats['interview']; ?></h3>
            </div>
            <div class="p-3 bg-primary-500/10 rounded-lg text-primary-500 group-hover:bg-primary-500 group-hover:text-white transition">
                <i class="fa-solid fa-headset text-xl"></i>
            </div>
        </div>
        <div class="absolute bottom-0 left-0 h-1 bg-primary-500 w-full opacity-50"></div>
    </a>

    <!-- Stat 3 -->
    <a href="whitelisted_users.php" class="glass-panel p-6 rounded-2xl relative overflow-hidden group hover:bg-dark-800 transition">
        <div class="flex justify-between items-start z-10 relative">
            <div>
                <p class="text-gray-400 text-xs font-bold uppercase tracking-widest">Whitelisted</p>
                <h3 class="text-3xl font-bold text-white mt-2"><?php echo $stats['accepted']; ?></h3>
            </div>
            <div class="p-3 bg-green-500/10 rounded-lg text-green-500 group-hover:bg-green-500 group-hover:text-white transition">
                <i class="fa-solid fa-check text-xl"></i>
            </div>
        </div>
        <div class="absolute bottom-0 left-0 h-1 bg-green-500 w-full opacity-50"></div>
    </a>

     <!-- Stat 4 -->
     <div class="glass-panel p-6 rounded-2xl relative overflow-hidden">
        <div class="flex justify-between items-start z-10 relative">
            <div>
                <p class="text-gray-400 text-xs font-bold uppercase tracking-widest">Declined</p>
                <h3 class="text-3xl font-bold text-white mt-2"><?php echo $stats['declined']; ?></h3>
            </div>
            <div class="p-3 bg-red-500/10 rounded-lg text-red-500">
                <i class="fa-solid fa-ban text-xl"></i>
            </div>
        </div>
        <div class="absolute bottom-0 left-0 h-1 bg-red-500 w-full opacity-50"></div>
    </div>
</div>

<!-- Main Content -->
<div class="glass-panel rounded-3xl overflow-hidden shadow-2xl animate-slide-up" style="animation-delay: 0.1s;">
    <div class="p-6 border-b border-dark-700 bg-dark-800/80 flex flex-col sm:flex-row justify-between items-center gap-4">
        <div class="flex items-center gap-3">
            <div class="w-10 h-10 rounded-full bg-yellow-500/10 flex items-center justify-center text-yellow-500">
                <i class="fa-solid fa-inbox"></i>
            </div>
            <h3 class="text-lg font-bold text-white">Pending Applications</h3>
        </div>
        <?php if (!empty($pending_apps)): ?>
            <span class="px-3 py-1 rounded-full bg-yellow-500 text-dark-900 text-xs font-bold uppercase animate-pulse">Action Required</span>
        <?php endif; ?>
    </div>

    <?php if ($action_message): ?>
        <div class="m-6 p-4 rounded-xl bg-green-500/10 border border-green-500/30 text-green-400 flex items-center">
            <i class="fa-solid fa-check-circle mr-2"></i> <?php echo htmlspecialchars($action_message); ?>
        </div>
    <?php endif; ?>

    <?php if (empty($pending_apps)): ?>
        <div class="p-20 text-center flex flex-col items-center">
            <div class="w-20 h-20 bg-dark-800 rounded-full flex items-center justify-center mb-6">
                <i class="fa-solid fa-clipboard-check text-3xl text-gray-600"></i>
            </div>
            <h3 class="text-xl font-bold text-white">All Clear</h3>
            <p class="text-gray-400 mt-2">No pending applications to review.</p>
        </div>
    <?php else: ?>
        <div class="overflow-x-auto">
            <table class="w-full text-left border-collapse">
                <thead>
                    <tr class="bg-dark-900/30 text-gray-500 text-xs uppercase tracking-wider border-b border-dark-700">
                        <th class="p-5 font-bold">Applicant</th>
                        <th class="p-5 font-bold">Submitted</th>
                        <th class="p-5 font-bold">Status</th>
                        <th class="p-5 font-bold text-right">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-dark-700/50 text-sm">
                    <?php foreach ($pending_apps as $app): ?>
                    <tr class="hover:bg-dark-700/30 transition-colors group">
                        <td class="p-5">
                            <div class="flex items-center gap-4">
                                <div class="w-10 h-10 rounded-lg bg-dark-800 border border-dark-700 flex items-center justify-center font-bold text-gray-400 group-hover:text-white transition">
                                    <?php echo strtoupper(substr($app['discord_username'], 0, 1)); ?>
                                </div>
                                <div>
                                    <div class="font-bold text-white"><?php echo htmlspecialchars($app['full_name']); ?></div>
                                    <div class="text-xs text-primary-400"><?php echo htmlspecialchars($app['discord_username']); ?></div>
                                </div>
                            </div>
                        </td>
                        <td class="p-5 text-gray-400">
                            <?php echo date('M j, g:i a', strtotime($app['submitted_at'])); ?>
                        </td>
                        <td class="p-5">
                            <span class="inline-flex items-center gap-1.5 px-2.5 py-1 rounded-lg text-xs font-bold bg-yellow-500/10 text-yellow-500 border border-yellow-500/20">
                                <span class="w-1.5 h-1.5 rounded-full bg-yellow-500"></span> Pending
                            </span>
                        </td>
                        <td class="p-5 text-right flex justify-end gap-2">
                            <!-- Global Modal Trigger -->
                            <button onclick='openViewModal(<?php echo json_encode($app); ?>)' class="px-3 py-2 bg-dark-800 hover:bg-dark-700 text-gray-300 rounded-lg transition border border-dark-700">
                                <i class="fa-solid fa-eye mr-2"></i> Review
                            </button>
                            
                            <form method="POST" class="inline" onsubmit="return confirm('Move to Interview Queue?');">
                                <input type="hidden" name="app_id" value="<?php echo $app['id']; ?>">
                                <input type="hidden" name="action" value="interview">
                                <button type="submit" class="px-3 py-2 bg-primary-600 hover:bg-primary-500 text-white rounded-lg transition shadow-lg shadow-primary-600/20">
                                    <i class="fa-solid fa-check"></i>
                                </button>
                            </form>
                            
                            <button onclick="showDeclineModal('<?php echo $app['id']; ?>', 'admin.php')" class="px-3 py-2 bg-red-500/10 hover:bg-red-500 text-red-500 hover:text-white rounded-lg transition border border-red-500/20">
                                <i class="fa-solid fa-xmark"></i>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<!-- Decline Modal -->
<div id="decline-modal" class="fixed inset-0 bg-black/80 backdrop-blur-sm hidden z-[110] flex items-center justify-center p-4">
    <div class="bg-dark-800 border border-dark-700 w-full max-w-md rounded-2xl shadow-2xl p-6 animate-slide-up">
        <h3 class="text-xl font-bold text-white mb-2">Decline Application</h3>
        <p class="text-gray-400 text-sm mb-6">Reason for rejection (sent to user):</p>
        <form method="POST" id="decline-form">
            <input type="hidden" name="app_id" id="decline-app-id" value="">
            <input type="hidden" name="action" value="decline">
            <textarea name="reason" rows="4" required class="w-full bg-dark-900 border border-dark-700 rounded-xl p-3 text-white focus:ring-2 focus:ring-red-500 outline-none mb-4 resize-none"></textarea>
            <div class="flex justify-end gap-3">
                <button type="button" onclick="document.getElementById('decline-modal').classList.add('hidden')" class="px-4 py-2 rounded-lg text-gray-400 hover:text-white hover:bg-dark-700 transition">Cancel</button>
                <button type="submit" class="px-4 py-2 bg-red-600 hover:bg-red-500 text-white rounded-lg font-bold transition">Confirm</button>
            </div>
        </form>
    </div>
</div>

<script>
    function showDeclineModal(appId, page) {
        document.getElementById('decline-app-id').value = appId;
        document.getElementById('decline-form').action = page;
        document.getElementById('decline-modal').classList.remove('hidden');
    }
</script>

<?php render_footer(); ?>