<?php
// application_handler.php - Handles form submissions and Admin actions
// Updated to support two-stage process (Interview -> Whitelist) with Discord Bot API.

// Load configuration
require_once 'config.php';

// --- DATA MANAGEMENT FUNCTIONS (DB Implementation) ---

function get_applications() {
    $pdo = get_db_connection();
    $stmt = $pdo->query("SELECT * FROM applications ORDER BY submitted_at DESC");
    return $stmt->fetchAll();
}

function get_application_by_id($app_id) {
    $pdo = get_db_connection();
    $stmt = $pdo->prepare("SELECT * FROM applications WHERE id = :id");
    $stmt->execute(['id' => $app_id]);
    return $stmt->fetch();
}


function handle_application_submission($post_data) {
    // ... (Submission logic remains the same) ...
    $db_data = $post_data;
    
    // Basic validation for required fields
    $required_fields = ['email', 'full_name', 'date_of_birth', 'steam_link', 'steam_profile_name', 'steam_hex_id', 'discord_username', 'discord_id', 'roleplay_explanation', 'new_life_rules_explanation', 'previous_roleplay_experience', 'read_rulebook', 'agree_to_rules'];

    foreach ($required_fields as $field) {
        if (empty($db_data[$field])) {
            return "Missing required field: $field";
        }
    }
    
    // Prepare data for insertion
    $db_data['id'] = uniqid('app_');
    $db_data['status'] = 'pending';
    $db_data['submitted_at'] = date('Y-m-d H:i:s');
    $db_data['discord_id'] = trim($db_data['discord_id']);
    
    // Ensure keys match DB structure
    $columns = implode(', ', array_keys($db_data));
    $placeholders = ':' . implode(', :', array_keys($db_data));
    
    try {
        $pdo = get_db_connection();
        $sql = "INSERT INTO applications ($columns) VALUES ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($db_data);
        return true;
    } catch (PDOException $e) {
        error_log("DB INSERT failed: " . $e->getMessage());
        return "Database error during submission.";
    }
}


// --- DISCORD API INTEGRATION (Role Management) ---

/**
 * Attempts to add a role to a member using the Discord Bot Token (PUT).
 */
function add_role_to_member($discord_id, $role_id) {
    if (empty(DISCORD_BOT_TOKEN) || empty(DISCORD_SERVER_ID) || empty($discord_id) || empty($role_id)) {
        return ['success' => false, 'message' => 'Discord Bot/Server/Role config missing.'];
    }

    $url = "https://discord.com/api/v10/guilds/" . DISCORD_SERVER_ID . "/members/" . $discord_id . "/roles/" . $role_id;
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bot ' . DISCORD_BOT_TOKEN,
        'Content-Length: 0',
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code == 204) {
        return ['success' => true, 'message' => 'Role added successfully.'];
    } else {
        $error_message = "Discord API Error ($http_code). Response: " . $response;
        error_log("Failed to add role to member: " . $error_message);
        return ['success' => false, 'message' => $error_message];
    }
}

/**
 * Attempts to remove a role from a member using the Discord Bot Token (DELETE).
 */
function remove_role_from_member($discord_id, $role_id) {
    if (empty(DISCORD_BOT_TOKEN) || empty(DISCORD_SERVER_ID) || empty($discord_id) || empty($role_id)) {
        return ['success' => false, 'message' => 'Discord config missing.'];
    }

    $url = "https://discord.com/api/v10/guilds/" . DISCORD_SERVER_ID . "/members/" . $discord_id . "/roles/" . $role_id;
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bot ' . DISCORD_BOT_TOKEN,
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    // 204 No Content is success for DELETE
    if ($http_code == 204 || $http_code == 404) { // Treat 404 (role not found) as success for removal
        return ['success' => true, 'message' => 'Role removed or already absent.'];
    } else {
        $error_message = "Discord API Error ($http_code) during role removal. Response: " . $response;
        error_log("Failed to remove role: " . $error_message);
        return ['success' => false, 'message' => $error_message];
    }
}

// --- DISCORD WEBHOOK INTEGRATION (Notification) ---

/**
 * Sends a webhook message with custom embeds and branding.
 */
function send_discord_webhook($embeds, $content = "") {
    if (empty(DISCORD_WEBHOOK_URL) || DISCORD_WEBHOOK_URL == 'YOUR_DISCORD_WEBHOOK_URL_HERE') {
        error_log("Discord Webhook URL is not configured or is placeholder.");
        return false;
    }

    $payload = [
        "content" => $content,
        "embeds" => $embeds,
    ];

    $ch = curl_init(DISCORD_WEBHOOK_URL);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-type: application/json'));
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        error_log("cURL error sending webhook: " . curl_error($ch));
    }
    curl_close($ch);

    if ($http_code >= 200 && $http_code < 300) {
        return true;
    } else {
        error_log("Failed to send Discord Webhook. HTTP Code: $http_code. Response: $response");
        return false;
    }
}

// --- ADMIN ACTION HANDLER (Three actions: interview, whitelist, decline) ---

function handle_application_action($app_id, $action, $reason = null) {
    $app = get_application_by_id($app_id);

    if (!$app) {
        return ['success' => false, 'message' => "Application not found."];
    }

    $pdo = get_db_connection();
    $discord_id = htmlspecialchars(trim($app['discord_id']));
    $username = htmlspecialchars($app['discord_username']);
    $update_stmt = $pdo->prepare("UPDATE applications SET status = :status WHERE id = :id");

    switch ($action) {
        case 'interview':
            $new_status = 'waiting_for_interview';
            
            // 1. Add WAITING role
            $role_result = add_role_to_member($discord_id, WAITING_ROLE_ID);

            // 2. Update DB Status
            $update_stmt->execute(['status' => $new_status, 'id' => $app_id]);

            // 3. Send Webhook (Initial Approval/Interview)
            $embed = [
                'author' => ['name' => WEBHOOK_AUTHOR_NAME, 'icon_url' => WEBHOOK_ICON_URL],
                'title' => "Whitelist Application Result",
                'description' => "Hi <@{$discord_id}>, your whitelist application has been **initially approved** ✅.\nYou've been given a **" . WAITING_ROLE_NAME . "** role. Please join a voice channel when staff is available to conduct your interview.\nThank you for your patience!",
                'color' => hexdec('FBC02D'), // Yellow/Amber
                'fields' => [['name' => 'Result', 'value' => 'Waiting For Interview ⏳', 'inline' => true]],
                'timestamp' => date('c'),
            ];
            send_discord_webhook([$embed], "<@{$discord_id}>");
            
            return $role_result['success'] ? 
                ['success' => true, 'message' => "Moved to Interview Queue. Role assigned."] :
                ['success' => false, 'message' => "Moved to Interview Queue, but Role assignment failed: " . $role_result['message']];


        case 'whitelist':
            $new_status = 'accepted';
            
            // 1. Remove WAITING role
            $remove_result = remove_role_from_member($discord_id, WAITING_ROLE_ID);
            
            // 2. Add ACCEPTED role
            $add_result = add_role_to_member($discord_id, ACCEPTED_ROLE_ID);

            // 3. Update DB Status
            $update_stmt->execute(['status' => $new_status, 'id' => $app_id]);

            // 4. Send Webhook (Final Acceptance)
            $embed = [
                'author' => ['name' => WEBHOOK_AUTHOR_NAME, 'icon_url' => WEBHOOK_ICON_URL],
                'title' => "Whitelist Application Result",
                'description' => "Hi <@{$discord_id}>, your whitelist application for Legion RP has been **approved** ✅.\nYou're now officially whitelisted and ready to join the RP experience!",
                'color' => hexdec('4CAF50'), // Green
                'fields' => [['name' => 'Result', 'value' => 'Passed ✅', 'inline' => true]],
                'timestamp' => date('c'),
            ];
            send_discord_webhook([$embed], "<@{$discord_id}>");
            
            return $add_result['success'] ? 
                ['success' => true, 'message' => "Application Whitelisted. Role assigned."] :
                ['success' => false, 'message' => "Whitelisted, but Role assignment failed: " . $add_result['message']];


        case 'decline':
            $new_status = 'declined';
            
            // 1. If currently in interview queue, remove WAITING role (if it exists)
            if ($app['status'] === 'waiting_for_interview') {
                remove_role_from_member($discord_id, WAITING_ROLE_ID);
            }
            
            // 2. Update DB Status
            $update_stmt->execute(['status' => $new_status, 'id' => $app_id]);
            
            // 3. Send Webhook (Rejection)
            $embed = [
                'author' => ['name' => WEBHOOK_AUTHOR_NAME, 'icon_url' => WEBHOOK_ICON_URL],
                'title' => "Whitelist Application Result",
                'description' => "Hi <@{$discord_id}>, your whitelist application has been **rejected** ❌.",
                'color' => hexdec('F44336'), // Red
                'fields' => [
                    ['name' => 'Result', 'value' => 'Failed ❌', 'inline' => false],
                    ['name' => 'Reason', 'value' => htmlspecialchars($reason ?? 'No reason provided.'), 'inline' => false],
                ],
                'timestamp' => date('c'),
            ];
            send_discord_webhook([$embed], "<@{$discord_id}>");
            
            return ['success' => true, 'message' => "Application Declined. User notified."];
            
        default:
            return ['success' => false, 'message' => "Unknown action."];
    }
}
?>