<?php
// layout.php - Modern UI Layout Manager & Global Scripts

function render_header($title = "Application System", $layout_type = 'public') {
    ?>
    <!DOCTYPE html>
    <html lang="en" class="dark">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title><?php echo htmlspecialchars($title); ?></title>
        
        <!-- Tailwind CSS -->
        <script src="https://cdn.tailwindcss.com"></script>
        
        <!-- FontAwesome -->
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        
        <!-- Google Fonts -->
        <link href="https://fonts.googleapis.com/css2?family=Space+Grotesk:wght@300;400;500;600;700&display=swap" rel="stylesheet">

        <script>
            tailwind.config = {
                darkMode: 'class',
                theme: {
                    extend: {
                        fontFamily: {
                            sans: ['Space Grotesk', 'sans-serif'],
                        },
                        colors: {
                            dark: {
                                900: '#0b0f19', // Deepest
                                800: '#111827', // Card BG
                                700: '#1f2937', // Border
                            },
                            primary: {
                                500: '#8b5cf6', // Violet
                                600: '#7c3aed',
                            },
                            accent: {
                                500: '#06b6d4', // Cyan
                            }
                        },
                        animation: {
                            'fade-in': 'fadeIn 0.5s ease-out',
                            'slide-up': 'slideUp 0.5s ease-out',
                        },
                        keyframes: {
                            fadeIn: {
                                '0%': { opacity: '0' },
                                '100%': { opacity: '1' },
                            },
                            slideUp: {
                                '0%': { transform: 'translateY(20px)', opacity: '0' },
                                '100%': { transform: 'translateY(0)', opacity: '1' },
                            }
                        }
                    }
                }
            }
        </script>

        <style>
            body { background-color: #0b0f19; color: #f8fafc; }
            .glass-panel {
                background: rgba(17, 24, 39, 0.7);
                backdrop-filter: blur(12px);
                border: 1px solid rgba(255, 255, 255, 0.08);
            }
            .sidebar-link.active {
                background: linear-gradient(to right, rgba(139, 92, 246, 0.2), transparent);
                border-left: 3px solid #8b5cf6;
                color: white;
            }
        </style>
    </head>
    <body class="antialiased min-h-screen flex flex-col md:flex-row overflow-hidden">

    <?php if ($layout_type === 'dashboard'): ?>
        <!-- Mobile Header -->
        <div class="md:hidden bg-dark-800 p-4 flex justify-between items-center border-b border-dark-700 z-50">
            <span class="font-bold text-xl text-primary-500 tracking-wider">NEXUS<span class="text-white">ADMIN</span></span>
            <button onclick="document.getElementById('sidebar').classList.toggle('-translate-x-full')" class="text-white">
                <i class="fa-solid fa-bars text-2xl"></i>
            </button>
        </div>

        <!-- Sidebar Navigation -->
        <nav id="sidebar" class="transform -translate-x-full md:translate-x-0 transition-transform duration-300 fixed md:static inset-y-0 left-0 z-40 w-72 bg-dark-800/90 backdrop-blur-xl border-r border-dark-700 flex flex-col h-screen">
            <div class="p-8 border-b border-dark-700/50">
                <h1 class="text-2xl font-bold text-white tracking-widest flex items-center gap-2">
                    <i class="fas fa-asterisk text-primary-500"></i>
                    ASTERISK
                </h1>
                <p class="text-xs text-gray-500 mt-1 uppercase tracking-widest">Administration</p>
            </div>

            <div class="flex-1 overflow-y-auto py-6 space-y-1">
                <p class="px-6 text-xs font-bold text-gray-500 uppercase mb-2">Main Menu</p>
                
                <a href="admin.php" class="sidebar-link <?php echo basename($_SERVER['PHP_SELF']) == 'admin.php' ? 'active' : 'text-gray-400 hover:text-white hover:bg-dark-700/50'; ?> flex items-center px-6 py-3 transition-all duration-200">
                    <i class="fa-solid fa-grid-2 w-6 text-center mr-3"></i> Dashboard
                </a>
                
                <a href="interview_queue.php" class="sidebar-link <?php echo basename($_SERVER['PHP_SELF']) == 'interview_queue.php' ? 'active' : 'text-gray-400 hover:text-white hover:bg-dark-700/50'; ?> flex items-center px-6 py-3 transition-all duration-200">
                    <i class="fa-solid fa-headset w-6 text-center mr-3"></i> Interview Queue
                </a>

                <a href="whitelisted_users.php" class="sidebar-link <?php echo basename($_SERVER['PHP_SELF']) == 'whitelisted_users.php' ? 'active' : 'text-gray-400 hover:text-white hover:bg-dark-700/50'; ?> flex items-center px-6 py-3 transition-all duration-200">
                    <i class="fa-solid fa-users-viewfinder w-6 text-center mr-3"></i> Whitelisted Members
                </a>

                <div class="mt-8">
                    <p class="px-6 text-xs font-bold text-gray-500 uppercase mb-2">System</p>
                    <a href="application_form.php" target="_blank" class="text-gray-400 hover:text-white hover:bg-dark-700/50 flex items-center px-6 py-3 transition-all">
                        <i class="fa-solid fa-arrow-up-right-from-square w-6 text-center mr-3"></i> View Live Form
                    </a>
                </div>
            </div>

            <div class="p-4 border-t border-dark-700/50">
                <a href="admin.php?logout" class="flex items-center justify-center w-full bg-red-500/10 text-red-400 hover:bg-red-500 hover:text-white transition p-3 rounded-xl font-medium border border-red-500/20">
                    <i class="fa-solid fa-power-off mr-2"></i> Sign Out
                </a>
            </div>
        </nav>
    <?php endif; ?>

    <!-- Main Content Area -->
    <main class="flex-1 overflow-x-hidden overflow-y-auto relative bg-dark-900 scroll-smooth">
        <!-- Ambient Background -->
        <div class="fixed top-0 left-0 w-full h-full overflow-hidden pointer-events-none z-0">
            <div class="absolute top-0 right-0 w-[500px] h-[500px] bg-primary-600/10 rounded-full blur-[100px] -translate-y-1/2 translate-x-1/2"></div>
            <div class="absolute bottom-0 left-0 w-[400px] h-[400px] bg-accent-500/5 rounded-full blur-[80px] translate-y-1/2 -translate-x-1/2"></div>
        </div>

        <div class="relative z-10 p-6 md:p-10 max-w-[1600px] mx-auto">
    <?php
}

function render_footer() {
    ?>
        </div>
    </main>

    <!-- Global View Application Modal -->
    <div id="view-modal" class="fixed inset-0 z-[100] hidden" aria-labelledby="modal-title" role="dialog" aria-modal="true">
        <!-- Backdrop -->
        <div class="fixed inset-0 bg-black/80 backdrop-blur-sm transition-opacity opacity-0" id="view-modal-backdrop"></div>

        <div class="fixed inset-0 z-10 overflow-y-auto">
            <div class="flex min-h-full items-center justify-center p-4 text-center sm:p-0">
                <div class="relative transform overflow-hidden rounded-2xl bg-dark-800 border border-dark-700 text-left shadow-2xl transition-all sm:my-8 sm:w-full sm:max-w-4xl opacity-0 translate-y-4 scale-95" id="view-modal-panel">
                    
                    <!-- Header -->
                    <div class="bg-dark-900/50 px-6 py-4 border-b border-dark-700 flex justify-between items-center">
                        <div>
                            <h3 class="text-xl font-bold text-white flex items-center gap-2" id="modal-title">
                                <i class="fa-regular fa-id-card text-primary-500"></i> Application Details
                            </h3>
                            <p class="text-sm text-gray-400 mt-1" id="modal-subtitle">Full overview</p>
                        </div>
                        <button onclick="closeViewModal()" class="text-gray-400 hover:text-white transition bg-dark-700 hover:bg-dark-600 rounded-lg p-2">
                            <i class="fa-solid fa-xmark text-lg"></i>
                        </button>
                    </div>

                    <!-- Content (Grid Layout) -->
                    <div class="px-6 py-6 max-h-[70vh] overflow-y-auto custom-scrollbar">
                        <div id="modal-content" class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                            <!-- Populated by JS -->
                        </div>
                    </div>

                    <!-- Footer -->
                    <div class="bg-dark-900/50 px-6 py-4 border-t border-dark-700 flex justify-end gap-3" id="modal-actions">
                        <button type="button" onclick="closeViewModal()" class="btn-secondary px-5 py-2.5 rounded-xl border border-dark-600 text-gray-300 hover:text-white hover:bg-dark-700 transition">Close</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Modal Logic
        const viewModal = document.getElementById('view-modal');
        const viewBackdrop = document.getElementById('view-modal-backdrop');
        const viewPanel = document.getElementById('view-modal-panel');
        const modalContent = document.getElementById('modal-content');

        function openViewModal(data) {
            viewModal.classList.remove('hidden');
            // Animation formatting
            setTimeout(() => {
                viewBackdrop.classList.remove('opacity-0');
                viewPanel.classList.remove('opacity-0', 'translate-y-4', 'scale-95');
            }, 10);

            // Populate Content
            let html = '';
            
            // Helper to format keys nicely
            const formatKey = (str) => {
                return str.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
            };

            // Define specific field order/groups if desired, otherwise iterate
            // Priority Fields
            const priority = ['full_name', 'discord_username', 'discord_id', 'status', 'email', 'date_of_birth'];
            const ignored = ['id', 'updated_at', 'agree_to_rules', 'read_rulebook']; // Skip technical/redundant fields

            // 1. Personal Info Block
            html += `<div class="col-span-1 lg:col-span-2 text-primary-400 text-xs font-bold uppercase tracking-widest mb-1 border-b border-dark-700 pb-2">User Identity</div>`;
            
            priority.forEach(key => {
                if(data[key]) {
                    html += `
                        <div class="bg-dark-900 p-4 rounded-xl border border-dark-700">
                            <span class="text-xs text-gray-500 uppercase block mb-1">${formatKey(key)}</span>
                            <span class="text-white font-medium break-words">${data[key]}</span>
                        </div>`;
                }
            });

            // 2. Gaming/Steam Block
            html += `<div class="col-span-1 lg:col-span-2 text-accent-500 text-xs font-bold uppercase tracking-widest mt-4 mb-1 border-b border-dark-700 pb-2">Gaming Accounts</div>`;
            ['steam_profile_name', 'steam_hex_id', 'steam_link'].forEach(key => {
                 if(data[key]) {
                    let val = data[key];
                    if(key === 'steam_link') val = `<a href="${val}" target="_blank" class="text-blue-400 hover:underline break-all">${val}</a>`;
                    
                    html += `
                        <div class="bg-dark-900 p-4 rounded-xl border border-dark-700">
                            <span class="text-xs text-gray-500 uppercase block mb-1">${formatKey(key)}</span>
                            <span class="text-white font-medium break-words font-mono">${val}</span>
                        </div>`;
                }
            });

            // 3. Long Form Answers
            html += `<div class="col-span-1 lg:col-span-2 text-green-500 text-xs font-bold uppercase tracking-widest mt-4 mb-1 border-b border-dark-700 pb-2">Application Answers</div>`;
            
            // Render remaining fields that are text areas
            for (const [key, value] of Object.entries(data)) {
                if (!priority.includes(key) && !ignored.includes(key) && !['steam_profile_name', 'steam_hex_id', 'steam_link'].includes(key)) {
                     html += `
                        <div class="col-span-1 lg:col-span-2 bg-dark-900 p-4 rounded-xl border border-dark-700">
                            <span class="text-xs text-gray-500 uppercase block mb-1">${formatKey(key)}</span>
                            <div class="text-gray-300 leading-relaxed whitespace-pre-wrap">${value}</div>
                        </div>`;
                }
            }

            modalContent.innerHTML = html;
        }

        function closeViewModal() {
            viewBackdrop.classList.add('opacity-0');
            viewPanel.classList.add('opacity-0', 'translate-y-4', 'scale-95');
            setTimeout(() => {
                viewModal.classList.add('hidden');
            }, 300);
        }
    </script>
    </body>
    </html>
    <?php
}
?>