<?php
// whitelisted_users.php - dedicated page for accepted members
require_once 'config.php';
require_once 'layout.php';
require_once 'application_handler.php';

session_start();

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin.php');
    exit;
}

// Fetch only accepted users
$applications = get_applications();
$whitelisted = array_filter($applications, function($app) {
    return $app['status'] === 'accepted';
});

render_header("Whitelisted Members", 'dashboard');
?>

<div class="flex flex-col md:flex-row justify-between items-center mb-8 gap-4 animate-fade-in">
    <div>
        <h2 class="text-3xl font-bold text-white tracking-tight">Whitelisted Members</h2>
        <p class="text-gray-400 mt-1">Directory of all officially accepted citizens.</p>
    </div>
    
    <div class="relative w-full md:w-96">
        <span class="absolute inset-y-0 left-0 pl-3 flex items-center text-gray-500"><i class="fa-solid fa-search"></i></span>
        <input type="text" id="searchInput" onkeyup="filterUsers()" placeholder="Search by name, discord, or steam..." class="w-full bg-dark-800 border border-dark-700 text-white rounded-xl pl-10 py-3 focus:outline-none focus:border-primary-500 focus:ring-1 focus:ring-primary-500 transition shadow-lg">
    </div>
</div>

<?php if (empty($whitelisted)): ?>
    <div class="text-center py-20 bg-dark-800/50 rounded-3xl border border-dark-700 border-dashed">
        <i class="fa-solid fa-users-slash text-6xl text-dark-600 mb-4"></i>
        <h3 class="text-xl text-gray-400 font-medium">No whitelisted members yet.</h3>
    </div>
<?php else: ?>
    <div class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-6" id="userGrid">
        <?php foreach ($whitelisted as $user): ?>
            <!-- User Card -->
            <div class="user-card glass-panel rounded-2xl p-6 relative group hover:border-primary-500/50 transition-all duration-300 hover:-translate-y-1">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex items-center gap-4">
                        <div class="w-14 h-14 rounded-full bg-gradient-to-br from-green-500 to-emerald-700 flex items-center justify-center text-xl font-bold text-white shadow-lg border-2 border-dark-800">
                            <?php echo strtoupper(substr($user['discord_username'], 0, 1)); ?>
                        </div>
                        <div>
                            <h4 class="text-lg font-bold text-white leading-tight"><?php echo htmlspecialchars($user['full_name']); ?></h4>
                            <span class="text-xs text-primary-400 bg-primary-500/10 px-2 py-0.5 rounded border border-primary-500/20 mt-1 inline-block">
                                <?php echo htmlspecialchars($user['discord_username']); ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- View Button -->
                    <button onclick='openViewModal(<?php echo json_encode($user); ?>)' class="text-gray-400 hover:text-white bg-dark-800 hover:bg-dark-700 p-2 rounded-lg transition">
                        <i class="fa-solid fa-expand"></i>
                    </button>
                </div>

                <div class="space-y-3 pt-4 border-t border-dark-700/50">
                    <div class="flex justify-between items-center text-sm">
                        <span class="text-gray-500"><i class="fa-brands fa-discord mr-2"></i>Discord ID</span>
                        <span class="text-gray-300 font-mono text-xs"><?php echo htmlspecialchars($user['discord_id']); ?></span>
                    </div>
                    <div class="flex justify-between items-center text-sm">
                        <span class="text-gray-500"><i class="fa-brands fa-steam mr-2"></i>Steam Hex</span>
                        <span class="text-gray-300 font-mono text-xs truncate max-w-[150px]"><?php echo htmlspecialchars($user['steam_hex_id']); ?></span>
                    </div>
                    <div class="flex justify-between items-center text-sm">
                        <span class="text-gray-500"><i class="fa-regular fa-calendar-check mr-2"></i>Accepted</span>
                        <span class="text-green-400 text-xs font-medium">Verified</span>
                    </div>
                </div>
                
                <!-- Hidden search data -->
                <span class="hidden search-data"><?php echo strtolower($user['full_name'] . ' ' . $user['discord_username'] . ' ' . $user['discord_id'] . ' ' . $user['steam_hex_id']); ?></span>
            </div>
        <?php endforeach; ?>
    </div>
<?php endif; ?>

<script>
function filterUsers() {
    const input = document.getElementById('searchInput');
    const filter = input.value.toLowerCase();
    const grid = document.getElementById('userGrid');
    const cards = grid.getElementsByClassName('user-card');

    for (let i = 0; i < cards.length; i++) {
        const searchData = cards[i].getElementsByClassName('search-data')[0].innerText;
        if (searchData.indexOf(filter) > -1) {
            cards[i].style.display = "";
        } else {
            cards[i].style.display = "none";
        }
    }
}
</script>

<?php render_footer(); ?>